//==============================================================================
// Project:		Wealth Transfers and their Economic Effects
// File name:	Wealth at Death - Expected Deaths by Marital Status - Excluding
//				Parents of Only Children
// Objective: 	Estimating mean and aggregate wealth at death for all persons
//				excluding parents of only children.
//
// Created: 	22/09/2021
// Modified: 	15/11/2021
//==============================================================================

**************************
*** Importing the Data ***
**************************

clear

cd "XXXX"	// set the content directory

use HILDA_restricted_combined_rescaled	// importing the combined dataset that contains all the different weights for nursing home residents

drop if tchave == 1

gen pw_net_worth = pwassei - pwdebti	// generating a personal net worth variable as the difference between personal assets and personal debts

gen own_home = 0
replace own_home = 1 if hstenr == 1

gen married = 0
replace married = 1 if mrcurr == 1

gen death_age = hgage
replace death_age = 100 if death_age > 100

keep xwaveid wave hhpxid pwobani pwjbani pwsupri pwsupwi pwoccdi pwjccdi pwhecdi pwothdi pwhmvai pwhmdti pwopvai pwopdti pwcbani pwcaini pweqini pwbusvi pwbusdi pwtrusi pwinsui pwcolli pwvechi pwobdti pwtbani pwsupei pwtpvi pwccdti pwtpdi pwfini pwnfii pwassei pwdebti hgage hgsex benchmarked* pw_net_worth own_home death_age married	// keeping all relevant variables

destring xwaveid, replace
destring hhpxid, replace

**********************************
*** Merging in Death Rate Data ***
**********************************

gen year = wave + 2000	// generating a year variable to enable merging with the age and gender specific death rates

merge m:1 year death_age hgsex married using "Death_rates_by_marital_status.dta", nogen	// merging in the age and gender specific death rate data

*************************
*** Cleaning the Data ***
*************************

drop if (wave != 2 & wave != 6 & wave != 10 & wave != 14 & wave != 18)	//drops waves that don't have a wealth module

codebook pw_net_worth
replace pw_net_worth = 0 if pw_net_worth < 0	// replacing negative values of net worth with zero because negative wealth cannot be passed on

save HILDA_onlychild_wealth, replace	// saving the dataset
save HILDA_onlychild_partner_wealth, replace	// saving a duplicate of the dataset to be used later to match partner death rates

**************************************
*** Weighting the Wealth Variables ***
**************************************

gen dw_bm_new_hhwte_pw_net_worth = pw_net_worth * benchmarked_new_hhwte * deathrate	// benchmark-weighted aggregate wealth

gen benchmarked_new_hhwte_deathrate = deathrate * benchmarked_new_hhwte	// multiplying the death rate variable by the benchmarked population-level enumerated person weights that include residents of non-private dwellings

collapse(sum) `wealth_vars' dw* deathrate benchmarked_new_hhwte_deathrate, by(year)	// collapsing (summing) the weighted and unweighted wealth variables and their death-weighted counterparts as well as the weighted and unweighted death rates by year

save HILDA_onlychild_wealth_at_death, replace	// saving the big dataset

keep year dw_bm_new_hhwte_pw_net_worth deathrate benchmarked_new_hhwte_deathrate	// keeping variables that will be used for plotting

save HILDA_onlychild_estates, replace	// saving the small dataset

*****************************************
*** Importing the Partner Wealth Data ***
*****************************************

clear

use HILDA_onlychild_partner_wealth

keep year xwaveid hhpxid deathrate	// keeping relevant identifiers and the death rate variable

drop if hhpxid == .	// dropping observations for those that do not have partners
drop hhpxid	// dropping the partner identifier variable

rename xwaveid hhpxid	// renaming the individual identifier to be the partner identifier
rename deathrate partner_deathrate	// renaming the death rate variable to allow it to be merged with the main dataset

save HILDA_onlychild_partner_deathrates, replace	// saving the dataset

************************************************
*** Merging the Partner Death Rate Variables ***
************************************************

clear

use HILDA_onlychild_wealth

merge m:1 hhpxid year using HILDA_onlychild_partner_deathrates	// merging in the partner death rates

replace deathrate = deathrate * partner_deathrate if partner_deathrate != .	// for couples, replacing the likelihood of death as the likelihood that both members of a couple die in a given year
rename deathrate final_deathrate

replace own_home = 0.5 * own_home if hhpxid != .

**************************************
*** Weighting the Wealth Variables ***
**************************************

gen dw_final_bm_new_pw_net_worth = pw_net_worth * benchmarked_new_hhwte * final_deathrate	// benchmark-weighted final estate wealth

gen bm_new_hhwte_final_deathrate = final_deathrate * benchmarked_new_hhwte	// multiplying the death rate variable by the benchmarked population-level enumerated person weights that include residents of non-private dwellings

save Onlychild_WAD_expected_deaths_big_final, replace

collapse(sum) `wealth_vars' dw* final_deathrate bm_new_hhwte_final_deathrate, by(year)	// collapsing (summing) the weighted and unweighted final estate wealth variables and their death-weighted counterparts as well as the weighted and unweighted death rates by year

save HILDA_onlychild_wealth_at_death_final, replace	// saving the final estates dataset

keep year dw_final_bm_new_pw_net_worth final_deathrate bm_new_hhwte_final_deathrate	// keeping variables that will be used for plotting

save HILDA_onlychild_final_estates, replace	// saving the small final estates dataset

********************************************
*** Merging with the All Estates Dataset ***
********************************************

merge 1:1 year using HILDA_onlychild_estates, nogen	// merging in the all estates dataset

************************************
*** Scaling Single Person Deaths ***
************************************

gen single_death_percent = bm_new_hhwte_final_deathrate / benchmarked_new_hhwte_deathrate	// generating a variable for single deaths as a proportion of total deaths

tab year single_death_percent	// tabulating the single person death percentage

merge 1:1 year using Single_person_deaths, nogen	// merging in ABS data on single person deaths as a proportion of all deaths

gen scale_factor = abs_single_death_percent / single_death_percent	// calculating a scale factor to account for the under representation of single deaths in the HILDA dataset

save onlychild_scale_factor.dta, replace

local final_estate_vars dw_final_bm_new_pw_net_worth final_deathrate bm_new_hhwte_final_deathrate	// creating a local macro with all aggregate final estate variables and the single person death rates

foreach w in `final_estate_vars' {
	replace `w' = `w' * scale_factor
}	// scaling the final estate variables by the single person death percentage scaling factor

save HILDA_onlychild_all_estates, replace	// saving the combined dataset (w property)

keep year dw_final_bm_new_pw_net_worth bm_new_hhwte_final_deathrate dw_bm_new_hhwte_pw_net_worth benchmarked_new_hhwte_deathrate single_death_percent abs_single_death_percent scale_factor